/*
 * stateflow.c
 *
 *  Created on: Apr 8, 2015
 *      Author: Jordan
 */

#include "sys_common.h"
#include "stateflow.h"
#include "board_io.h"
#include "guidance.h"
#include "PWM.h"
#include "control.h"
#include "navigation.h"
#include "config.h"
#include "gps.h"

#include "sci.h"

autopilot_state_t autopilot_state = AUTOPILOT_DISARMED;
autopilot_state_t autopilot_next_state;
uint8 confirm_cmd = 0;

flight_mode_t flight_mode = FLIGHT_MODE_ANGLE;

stick_cmd_t arm_disarm = {STICK_MID, STICK_MID, STICK_LOW, STICK_LOW, STICK_DONT_CARE, STICK_DONT_CARE};
stick_cmd_t stick_neutral = {STICK_MID, STICK_MID, STICK_MID, STICK_LOW, STICK_DONT_CARE, STICK_DONT_CARE};
stick_cmd_t tuning_mode = {STICK_MID, STICK_MID, STICK_HIGH, STICK_LOW, STICK_DONT_CARE, STICK_DONT_CARE};

autopilot_state_t get_autopilot_state(){
	return autopilot_state;
}

void update_state(){
	float32 ail = get_ail_cmd();
	float32 ele = get_ele_cmd();
	float32 rud = get_rud_cmd();
	float32 thr = get_thr_cmd();
	float32 aux1 = get_PWM_width(5);
	float32 aux2 = get_PWM_width(6);
	float32 stick_array[6] = {ail, ele, rud, thr, aux1, aux2};

	if(!confirm_cmd){
		if(autopilot_state == AUTOPILOT_ARMED){
			if(check_stick_cmd(&arm_disarm, stick_array)){
				autopilot_next_state = AUTOPILOT_DISARMED;
				confirm_cmd = 1;
			}
		}
		if(autopilot_state == AUTOPILOT_TUNING){
			if(check_stick_cmd(&tuning_mode, stick_array)){
				autopilot_next_state = AUTOPILOT_DISARMED;
				confirm_cmd = 1;
			}
			float32 change = 0;
			if(aux1 > 1.5)
			{
				if(thr < 1.3){ // PID - P
					if(ele < 1 + CMD_THRESHOLD){ // Increase attitude rate P
						adjust_PID(ROLL_RATE_P, RATE_P_INC);
						adjust_PID(PITCH_RATE_P, RATE_P_INC);
						change = 1;
					} else if(ele > 2 - CMD_THRESHOLD){ // Decrease attitude rate P
						adjust_PID(ROLL_RATE_P, -RATE_P_INC);
						adjust_PID(PITCH_RATE_P, -RATE_P_INC);
						change = 1;
					} else if(ail > 2 - CMD_THRESHOLD){ // Increase yaw rate P
						adjust_PID(YAW_RATE_P, RATE_P_INC);
						change = 1;
					} else if(ail < 1 + CMD_THRESHOLD){ // Decrease yaw rate P
						adjust_PID(YAW_RATE_P, -RATE_P_INC);
						change = 1;
					}
				} else if(thr > 1.7){ // PID - D
					if(ele < 1 + CMD_THRESHOLD){ // Increase attitude rate D
						adjust_PID(ROLL_RATE_D, RATE_D_INC);
						adjust_PID(PITCH_RATE_D, RATE_D_INC);
						change = 1;
					} else if(ele > 2 - CMD_THRESHOLD){ // Decrease attitude rate D
						adjust_PID(ROLL_RATE_D, -RATE_D_INC);
						adjust_PID(PITCH_RATE_D, -RATE_D_INC);
						change = 1;
					} else if(ail > 2 - CMD_THRESHOLD){ // Increase yaw rate D
						adjust_PID(YAW_RATE_D, RATE_D_INC);
						change = 1;
					} else if(ail < 1 + CMD_THRESHOLD){ // Decrease yaw rate D
						adjust_PID(YAW_RATE_D, -RATE_D_INC);
						change = 1;
					}
				} else { // PID - I
					if(ele < 1 + CMD_THRESHOLD){ // Increase attitude rate I
						adjust_PID(ROLL_RATE_I, RATE_I_INC);
						adjust_PID(PITCH_RATE_I, RATE_I_INC);
						change = 1;
					} else if(ele > 2 - CMD_THRESHOLD){ // Decrease attitude rate I
						adjust_PID(ROLL_RATE_I, -RATE_I_INC);
						adjust_PID(PITCH_RATE_I, -RATE_I_INC);
						change = 1;
					} else if(ail > 2 - CMD_THRESHOLD){ // Increase yaw rate I
						adjust_PID(YAW_RATE_I, RATE_I_INC);
						change = 1;
					} else if(ail < 1 + CMD_THRESHOLD){ // Decrease yaw rate I
						adjust_PID(YAW_RATE_I, -RATE_I_INC);
						change = 1;
					}
				}
			} else {
				if(thr < 1.3){ // PID - P
					if(ele < 1 + CMD_THRESHOLD){ // Increase attitude angle P
						adjust_PID(ROLL_P, ANGLE_P_INC);
						adjust_PID(PITCH_P, ANGLE_P_INC);
						change = 1;
					} else if(ele > 2 - CMD_THRESHOLD){ // Decrease attitude angle P
						adjust_PID(ROLL_P, -ANGLE_P_INC);
						adjust_PID(PITCH_P, -ANGLE_P_INC);
						change = 1;
					}
				} else if(thr > 1.7){ // PID - D
					if(ele < 1 + CMD_THRESHOLD){ // Increase attitude angle D
						adjust_PID(ROLL_D, ANGLE_D_INC);
						adjust_PID(PITCH_D, ANGLE_D_INC);
						change = 1;
					} else if(ele > 2 - CMD_THRESHOLD){ // Decrease attitude angle D
						adjust_PID(ROLL_D, -ANGLE_D_INC);
						adjust_PID(PITCH_D, -ANGLE_D_INC);
						change = 1;
					}
				} else { // PID - I
					if(ele < 1 + CMD_THRESHOLD){ // Increase attitude angle I
						adjust_PID(ROLL_I, ANGLE_I_INC);
						adjust_PID(PITCH_I, ANGLE_I_INC);
						change = 1;
					} else if(ele > 2 - CMD_THRESHOLD){ // Decrease attitude angle I
						adjust_PID(ROLL_I, -ANGLE_I_INC);
						adjust_PID(PITCH_I, -ANGLE_I_INC);
						change = 1;
					}
				}
			}
			if(change){
				#if DISPLAY_PID == 1
					print_PID();
				#endif
				update_LEDs(2,0,2,2);
				delay_ms(250);
				update_LEDs(2,1,2,2);
				delay_ms(250);
			}
		}
		if(autopilot_state == AUTOPILOT_DISARMED){
			if(check_stick_cmd(&arm_disarm, stick_array)){
				if(fabs(get_roll()) < PI/4 && fabs(get_pitch()) < PI/4){
					INS_init();
					if(flight_mode == FLIGHT_MODE_VELOCITY || flight_mode == FLIGHT_MODE_WAYPOINT){
						if(get_INS_enabled()){
							autopilot_next_state = AUTOPILOT_ARMED;
							confirm_cmd = 1;
						} else {
							update_LEDs(1,2,2,2);
							delay_ms(250);
							update_LEDs(0,2,2,2);
							delay_ms(250);
						}
					} else {
						autopilot_next_state = AUTOPILOT_ARMED;
						confirm_cmd = 1;
					}
				}
			}
			if(check_stick_cmd(&tuning_mode, stick_array)){
				autopilot_next_state = AUTOPILOT_TUNING;
				confirm_cmd = 1;
			}
		}
	} else {
		if(check_stick_cmd(&stick_neutral, stick_array)){
			confirm_cmd = 0;

			if(autopilot_state == AUTOPILOT_TUNING){
				write_PID_params();
			}

			autopilot_state = autopilot_next_state;
			switch(autopilot_state){
				case AUTOPILOT_DISARMED:
					update_LEDs(0,0,2,2);
					break;

				case AUTOPILOT_ARMED:
					update_LEDs(1,2,2,2);
					AHRS_init();
					control_reset();
					break;

				case AUTOPILOT_TUNING:
//					sciSetBaudrate(scilinREG, 57600);
//					sciSetBaudrate(sciREG, 57600);
					update_LEDs(2,1,2,2);
					#if DISPLAY_PID == 1
						print_PID();
					#endif
					break;
			}
		}
	}

	if(aux1 > 1.66){
//		if(flight_mode != FLIGHT_MODE_RATE){
//			flight_mode = FLIGHT_MODE_RATE;
//			update_LEDs(2,2,0,1);
//			if(autopilot_state == AUTOPILOT_ARMED){
//				control_reset();
//			}
//		}
		if(flight_mode != FLIGHT_MODE_ANGLE){
			flight_mode = FLIGHT_MODE_ANGLE;
			update_LEDs(2,2,0,0);
			if(autopilot_state == AUTOPILOT_ARMED){
				control_reset();
			}
		}
	} else if(aux1 > 1.33){
		if(autopilot_state == AUTOPILOT_ARMED){
			if(get_INS_enabled()){
				if(flight_mode != FLIGHT_MODE_VELOCITY){
					flight_mode = FLIGHT_MODE_VELOCITY;
					update_LEDs(2,2,1,1);
					control_reset();
				}
			}
		} else {
			flight_mode = FLIGHT_MODE_VELOCITY;
			update_LEDs(2,2,1,1);
		}
	} else {
		if(autopilot_state == AUTOPILOT_ARMED){
			if(get_INS_enabled()){
				if(flight_mode != FLIGHT_MODE_WAYPOINT){
					flight_mode = FLIGHT_MODE_WAYPOINT;
					update_LEDs(2,2,1,0);
					control_reset();
				}
			}
		} else {
			flight_mode = FLIGHT_MODE_WAYPOINT;
			update_LEDs(2,2,1,0);
		}
	}
}

uint8 check_stick_cmd(stick_cmd_t* stick_cmd, float32* stick_array){
	float32 cmd_array[6] = {stick_cmd->ail, stick_cmd->ele, stick_cmd->rud, stick_cmd->thr, stick_cmd->aux1, stick_cmd->aux2};
	uint16 i;
	uint8 result = 1;
	for(i=0;i<6;i++){
		if(cmd_array[i] == STICK_LOW && stick_array[i] > 1 + CMD_THRESHOLD){
			result = 0;
		} else if(cmd_array[i] == STICK_MID && fabs(stick_array[i] - 1.5) > CMD_THRESHOLD){
			result = 0;
		} else if(cmd_array[i] == STICK_HIGH && stick_array[i] < 2 - CMD_THRESHOLD){
			result = 0;
		}
	}
	return result;
}

flight_mode_t get_flight_mode(){
	return flight_mode;
}
